<?php
/* ===== CONFIG (edit these) ===== */
const CLIENT_EMAILS = 'brightonpaintersuk@gmail.com';                 // who gets the leads
const ADMIN_EMAIL   = 'emails@miracleventures.co.uk';        // your copy
const STORAGE_DIR   = __DIR__ . '/_lead_store';              // will be created
const SITE_NAME     = 'New Adwords Lead';               // appears in subject
const FROM_EMAIL    = 'emails@miracleventures.co.uk';        // sender for mail()

/* ==== SMS via ClickSend ==== */
const CS_SMS_ENABLED = true;                                 // flip to false to disable SMS
const CS_USERNAME    = 'emails@miracleventures.co.uk';       // ClickSend username (often your email)
const CS_API_KEY     = '6B5BEFC1-2FB1-E91F-B56F-BBBBB4C90017'; // ClickSend API key
const CS_SENDER      = '';                                   // leave blank so ClickSend uses a compliant numeric sender
const CS_RECIPIENTS  = '+447525098073';                      // comma-separated list of mobiles in E.164 (+44...)
/* =============================== */

/* ===== DEBUG (temporary; set to false in production) ===== */
const DEBUG_ENABLED = true;
const SELFTEST_SECRET = 'CHANGE_ME'; // for GET self-tests; change or set empty to disable
function dbg($label, $data = null) {
  if (!DEBUG_ENABLED) return;
  @file_put_contents(
    STORAGE_DIR.'/debug.log',
    sprintf("[%s] %s %s\n", date('c'), $label, $data !== null ? json_encode($data, JSON_UNESCAPED_UNICODE) : ''),
    FILE_APPEND
  );
}

/* ===== Optional self-test (visit /lead.php?selftest=1&secret=CHANGE_ME) ===== */
if (!empty($_GET['selftest']) && $_GET['selftest'] === '1' && SELFTEST_SECRET && (($_GET['secret'] ?? '') === SELFTEST_SECRET)) {
  $_POST = [
    'name'        => 'Self Test',
    'phone'       => '07900123456',
    'email'       => '',
    'message'     => 'Ping from self-test',
    'service'     => 'Kitchen Respraying',   // <-- include service in self-test
    'utm_campaign'=> 'selftest',
    'return_url'  => (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? 'https://' : 'http://') .
                     ($_SERVER['HTTP_HOST'] ?? 'localhost') . '/thank-you/'
  ];
  $_SERVER['REQUEST_METHOD'] = 'POST';
  dbg('SELFTEST triggered');
}

/* ===== CORS (only needed for cross-domain AJAX) ===== */
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(200); exit; }

/* ===== Method guard ===== */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') { http_response_code(405); echo 'Method not allowed'; exit; }

/* ===== Parse POST (JSON or form-encoded) ===== */
$raw = file_get_contents('php://input');
$ct  = $_SERVER['CONTENT_TYPE'] ?? '';
$data = [];
if ($raw && stripos($ct, 'application/json') !== false) {
  $tmp = json_decode($raw, true);
  if (json_last_error() === JSON_ERROR_NONE) $data = $tmp;
}
if (!$data) $data = $_POST;

dbg('POST received', [
  'method' => $_SERVER['REQUEST_METHOD'] ?? '',
  'ct'     => $_SERVER['CONTENT_TYPE'] ?? '',
  'keys'   => array_keys($data)
]);

/* ===== Honeypot (safe to ignore if you removed it on the form) ===== */
dbg('honeypot', ['hp' => ($data['hp'] ?? null)]);
if (!empty($data['hp'])) {
  header('Content-Type: application/json; charset=utf-8');
  dbg('honeypot_block', $data['hp']);
  echo json_encode(['ok'=>true, 'ignored'=>'honeypot']); exit;
}

/* ===== Extract fields ===== */
$name     = trim($data['name']    ?? '');
$email    = trim($data['email']   ?? '');
$phone    = trim($data['phone']   ?? '');
$message  = trim($data['message'] ?? '');

/* NEW: service (with fallback to template filename) */
$service   = trim($data['service'] ?? '');
$template  = trim($data['template_filename'] ?? '');
if ($service === '' && $template !== '') {
  $service = ucwords(str_replace(['-', '_'], ' ', $template));
}

$gclid    = trim($data['gclid']   ?? '');
$utm_s    = trim($data['utm_source']   ?? '');
$utm_m    = trim($data['utm_medium']   ?? '');
$utm_c    = trim($data['utm_campaign'] ?? '');
$thanks   = filter_var($data['return_url'] ?? '', FILTER_VALIDATE_URL);

$ip  = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
$ua  = $_SERVER['HTTP_USER_AGENT'] ?? '';
$ts  = date('c');
$eid = substr(hash('sha256', implode('|', [$name,$email,$phone,$message,$service,$gclid,date('Y-m-d')])), 0, 16);

dbg('fields_parsed', ['name'=>$name,'email'=>$email,'phone'=>$phone,'service'=>$service,'thanks'=>$thanks,'utm_c'=>$utm_c]);

/* ===== Ensure storage dir exists & is protected ===== */
if (!is_dir(STORAGE_DIR)) {
  $mk = @mkdir(STORAGE_DIR, 0755, true);
  dbg('mkdir', ['ok'=>$mk, 'dir'=>STORAGE_DIR, 'last_error'=>error_get_last()]);
}
$ht = STORAGE_DIR.'/.htaccess';
if (!file_exists($ht)) { @file_put_contents($ht, "Deny from all\n"); }

/* ===== Persist to JSONL (append-only) ===== */
$record = [
  'timestamp'=>$ts,'name'=>$name,'email'=>$email,'phone'=>$phone,'message'=>$message,
  'service'=>$service, 'template'=>$template,                              // <-- store service + template
  'gclid'=>$gclid,'utm_source'=>$utm_s,'utm_medium'=>$utm_m,'utm_campaign'=>$utm_c,
  'ip'=>$ip,'user_agent'=>$ua,'event_id'=>$eid
];
dbg('about_to_write', ['path'=>STORAGE_DIR.'/leads.jsonl']);
$bytes = @file_put_contents(
  STORAGE_DIR.'/leads.jsonl',
  json_encode($record, JSON_UNESCAPED_UNICODE) . PHP_EOL,
  FILE_APPEND | LOCK_EX
);
dbg('write_result', ['bytes'=>$bytes, 'writable_dir'=>is_writable(STORAGE_DIR), 'last_error'=>error_get_last()]);

/* ===== Email notifications (uses envelope sender for SPF/DMARC) ===== */
$who     = $name ?: ($phone ?: ($email ?: 'Enquiry'));
$subject = "New Lead – ".SITE_NAME." – ".$who;

$lines = [
  "New enquiry on ".SITE_NAME,
  "",
  "Name: $name",
  "Email: $email",
  "Phone: $phone",
  "Service: " . ($service !== '' ? $service : 'General'),   // <-- include service
  "",
  "Message:",
  $message,
  "",
  "Attribution:",
  "gclid: $gclid",
  "utm_source: $utm_s",
  "utm_medium: $utm_m",
  "utm_campaign: $utm_c",
  "",
  "Event ID: $eid",
  "IP: $ip",
  "User-Agent: $ua",
];
$body = implode("\n", $lines);

$headers   = [];
$headers[] = 'From: Website Leads <'.FROM_EMAIL.'>';
$headers[] = 'Reply-To: '.($email ?: FROM_EMAIL); // or set to FROM_EMAIL to keep replies with you
$headers[] = 'MIME-Version: 1.0';
$headers[] = 'Content-Type: text/plain; charset=UTF-8';
$headers[] = 'X-Mailer: PHP/'.phpversion();
$headersStr = implode("\r\n", $headers);
$extraParams = '-f '.FROM_EMAIL; // envelope sender

$okClient = @mail(CLIENT_EMAILS, mb_encode_mimeheader($subject,'UTF-8'), $body, $headersStr, $extraParams);
$okAdmin  = @mail(ADMIN_EMAIL,   mb_encode_mimeheader($subject,'UTF-8'), $body, $headersStr, $extraParams);
dbg('email_results', ['client_ok'=>$okClient,'admin_ok'=>$okAdmin]);

if (!$okClient || !$okAdmin) {
  @file_put_contents(
    STORAGE_DIR.'/email_errors.log',
    sprintf("[%s] mail() failed; client=%s admin=%s\n", date('c'), var_export($okClient,true), var_export($okAdmin,true)),
    FILE_APPEND
  );
}

/* ===== SMS notifications (ClickSend) ===== */
if (CS_SMS_ENABLED && CS_USERNAME !== '' && CS_API_KEY !== '') {
  $recipients = array_values(array_unique(array_filter(array_map('trim', explode(',', CS_RECIPIENTS)))));
  if (!empty($recipients)) {
    $msgShort = preg_replace('/\s+/', ' ', (string)$message);
    $msgShort = mb_substr($msgShort, 0, 300);
    $smsBody  = "New lead: {$who}\n"
              . ($service ? "Service: $service\n" : '')          // <-- include service
              . ($phone ? "Tel: $phone\n"   : '')
              . ($email ? "Email: $email\n" : '')
              . ($utm_c  ? "Campaign: $utm_c\n" : '')
              . ($msgShort ? "Msg: $msgShort" : '');

    $smsRes = clicksend_send_sms_debug($recipients, $smsBody);
    dbg('sms_result', $smsRes);
    if (!$smsRes['ok']) {
      @file_put_contents(
        STORAGE_DIR.'/sms_errors.log',
        sprintf("[%s] ClickSend fail HTTP=%s resp=%s err=%s\n", date('c'), $smsRes['http'], $smsRes['resp'], $smsRes['curl_err']),
        FILE_APPEND
      );
    }
  } else {
    dbg('sms_skip', 'no_recipients');
  }
} else {
  dbg('sms_skip', 'disabled_or_missing_creds');
}

/* ===== Redirect or JSON ===== */
if ($thanks) { dbg('redirect', ['to'=>$thanks]); header('Location: '.$thanks, true, 302); exit; }
header('Content-Type: application/json; charset=utf-8');
echo json_encode(['ok'=>true, 'event_id'=>$eid]);

/* ===== Helpers ===== */

/**
 * Debug-capable ClickSend sender returning detailed result.
 * @param array  $numbers  E.164 numbers (e.g., ['+447...'])
 * @param string $body     Message text
 * @return array { ok:bool, http:int, resp:string, curl_err:string }
 */
function clicksend_send_sms_debug(array $numbers, string $body): array {
  $endpoint = 'https://rest.clicksend.com/v3/sms/send';
  $messages = [];
  foreach ($numbers as $to) {
    $msg = [
      'source' => 'php',
      'to'     => $to,
      'body'   => $body,
    ];
    if (CS_SENDER !== '') { $msg['from'] = CS_SENDER; }
    $messages[] = $msg;
  }
  $payload = json_encode(['messages' => $messages], JSON_UNESCAPED_UNICODE);

  $ch = curl_init($endpoint);
  curl_setopt_array($ch, [
    CURLOPT_POST           => true,
    CURLOPT_POSTFIELDS     => $payload,
    CURLOPT_HTTPHEADER     => ['Content-Type: application/json'],
    CURLOPT_USERPWD        => CS_USERNAME . ':' . CS_API_KEY, // Basic Auth
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT        => 20,
  ]);
  $resp = curl_exec($ch);
  $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
  $err  = curl_error($ch);
  curl_close($ch);

  $ok = false;
  if ($http >= 200 && $http < 300) {
    $j = json_decode($resp, true);
    // ClickSend "SUCCESS" when queued
    $ok = isset($j['response_code']) && strtoupper($j['response_code']) === 'SUCCESS';
  }
  return ['ok'=>$ok, 'http'=>$http, 'resp'=>$resp, 'curl_err'=>$err];
}
